package com.yutu.base.service;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.yutu.base.entity.Response;
import com.yutu.base.entity.SaleDetail;
import com.yutu.base.utils.HttpsSendData;
import com.yutu.comm.response.NoteResult;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.CommandLineRunner;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletResponse;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingDeque;


/**
 * 销售明细报表相关操作
 * @author pc
 * @date 20210324104930
 */
@Service
public class SaleDetailService implements CommandLineRunner {


    @Value("${spring.urlConfig.overseasAirTicketUrl}")
    private String overseasAirTicketUrl;

    @Value("${spring.urlConfig.statementApiUrl}")
    private String statementApiUrl;


    @Value("${spring.pathConfig.saleDetailReportExcelPath}")
    private String  saleDetailReportExcelPath;
    @Value("${spring.urlConfig.productServiceUrl}")
    private String productServiceUrl;

    @Autowired
    private ParseToSaleDetail parseToSaleDetail;

    private static Logger logger = Logger.getLogger("saleDetail");


    /**
     * 获取销售明细报表数据
     *      参数：
     *          startPurchaseTime endPurchaseTime 采购时间
     *          startCreateOrderTime endCreateOrderTime 生单时间
     *          startTicketTime endTicketTime 出票时间
     *
     * @param requestBody
     * @return
     */
    public NoteResult getSalesDetailList(String requestBody) {
        NoteResult noteResult = new NoteResult();
        if (StringUtils.isBlank(requestBody)) {
            noteResult.setStatus(300);
            noteResult.setMsg("参数不能为空 [requestBody]");
            return noteResult;
        }
        logger.info("开始获取数据，参数：" + requestBody);
        long beginTime = System.currentTimeMillis();
        // String result = HttpsSendData.getPost(overseasAirTicketUrl+"orderInfo/getReportExcelData.do",requestBody);    autoSaleDetai/findByKeyList

        String result = HttpsSendData.getPost(statementApiUrl + "saledetail/getSaleDetailList", requestBody);
        // String result = "{\"status\":0,\"msg\":\"查询成功\",\"data\":[{\"id\":\"00027727-2e16-485a-b1b3-f13d5079645a\",\"key\":\"df0286f1-eaac-11e7-adf4-a4badb3682ff_2018-10-25_2018-10-26\",\"createTime\":\"2018-12-21 09:41:41\",\"updateTime\":\"2018-12-21 09:41:41\",\"createOrderTime\":\"2018-10-24 16:24:26\",\"ticketTime\":\"2018-10-25 00:55:41\",\"outerOrderNo\":\"1380889117689\",\"channel\":\"Taobao\",\"store\":\"tbml\",\"tairLine\":\"HKT-DMK\",\"flightTime\":\"1544358000000\",\"flightNumber\":\"FD3028\",\"tripType\":\"单程\",\"pnrType\":\"去程\",\"pnr\":\"BWSTUZ\",\"passengerCount\":1,\"salePrice\":128,\"luggagePrice\":0,\"points\":0,\"purchaseCurrency\":\"CNY\",\"purchasePrice_local\":92,\"purchasePrice_rmb\":92,\"transactionDetail_local\":92,\"transactionDetail_rmb\":92,\"payMethod\":\"支付宝huochepiao3\",\"cardNumber\":\"支付宝huochepiao3\",\"profit\":36,\"transactionNumber\":\" \",\"orderValueAddedNo\":\" \",\"purchaseChannel\":\"艺龙\",\"purchaseAccount\":\"18513092060----huidarui\",\"purchaseOrderNo\":\"IT7JP5181024MO515179\",\"policyType\":\"普通促销\",\"policyCode\":\"TGJ泰国-中国\",\"orderStatus\":\"已出票\",\"operator\":\"曹坤祥\",\"purchaseTime\":\"2018-10-24 19:01:39\",\"orderValueAdded\":\" \",\"orderValueAddedSpecific\":\" \",\"remark\":\" \",\"officalReceivedPaymentStatus\":\" \",\"officalActualExpendAmount\":0,\"officalBalance\":0,\"officalRateAdjustDate\":null,\"passengerNames\":null,\"adultCount\":1,\"childCount\":0,\"sortIndex\":438}],\"info\":null}";

        if (StringUtils.isBlank(result)) {
//            return Response.error("获取数据失败");
            noteResult.setStatus(300);
            noteResult.setMsg("获取数据失败");
            noteResult.setData(new HashMap<String, String>());
            return noteResult;
        }
        JSONObject resultObj = JSONObject.parseObject(result);
        if (resultObj.getInteger("status") != 200) {
//            return Response.error("获取数据失败");
            noteResult.setMsg("获取数据失败");
            return noteResult;
        }
        logger.info("获取数据成功，共用时：" + (System.currentTimeMillis() - beginTime));
//        Map<String, String> resultMap = resultObj.getObject("data", Map.class);
//        noteResult.setStatus(200);

        // com.alibaba.fastjson.JSONException: can not cast to : java.util.Map
////        noteResult.setData(resultMap);
//        noteResult.setData(resultObj.getObject("data", NoteResult.class));      // "data":{}, 前端报错
//        noteResult.setMsg("获取成功");

        return  JSONObject.toJavaObject(resultObj, NoteResult.class);
    }

    /**
     * 获取销售明细报表数据，并将其处理为 Excel
     * @param requestBody
     * @param response
     * @return
     */
    public NoteResult downloadSalesDetailReportExcel(String requestBody, HttpServletResponse response){
        NoteResult noteResult = new NoteResult();


        if (StringUtils.isBlank(requestBody)) {

            noteResult.setStatus(300);
            noteResult.setMsg("参数不能为空 [requestBody]");
            return noteResult;
        }

        String keyList = JSONObject.parseObject(requestBody).getString("keyList");
        logger.info("[AutoSource] 从saleDetail表数据中直接下载报表，keyList="+keyList);
        long beginTime = System.currentTimeMillis();
        Map<String,String> params = new HashMap<>();
        params.put("keyList", keyList);

         String result = HttpsSendData.send(statementApiUrl + "saledetail/findByKeyList", params);
//        String result = HttpsSendData.getPost(statementApiUrl + "saledetail/getSaleDetailList", requestBody);

        // String result = "{\"status\":200,\"msg\":\"查询成功\",\"data\":[{\"id\":\"00027727-2e16-485a-b1b3-f13d5079645a\",\"key\":\"df0286f1-eaac-11e7-adf4-a4badb3682ff_2018-10-25_2018-10-26\",\"createTime\":\"2018-12-21 09:41:41\",\"updateTime\":\"2018-12-21 09:41:41\",\"createOrderTime\":\"2018-10-24 16:24:26\",\"ticketTime\":\"2018-10-25 00:55:41\",\"outerOrderNo\":\"1380889117689\",\"channel\":\"Taobao\",\"store\":\"tbml\",\"tairLine\":\"HKT-DMK\",\"flightTime\":\"1544358000000\",\"flightNumber\":\"FD3028\",\"tripType\":\"单程\",\"pnrType\":\"去程\",\"pnr\":\"BWSTUZ\",\"passengerCount\":1,\"salePrice\":128,\"luggagePrice\":0,\"points\":0,\"purchaseCurrency\":\"CNY\",\"purchasePrice_local\":92,\"purchasePrice_rmb\":92,\"transactionDetail_local\":92,\"transactionDetail_rmb\":92,\"payMethod\":\"支付宝huochepiao3\",\"cardNumber\":\"支付宝huochepiao3\",\"profit\":36,\"transactionNumber\":\" \",\"orderValueAddedNo\":\" \",\"purchaseChannel\":\"艺龙\",\"purchaseAccount\":\"18513092060----huidarui\",\"purchaseOrderNo\":\"IT7JP5181024MO515179\",\"policyType\":\"普通促销\",\"policyCode\":\"TGJ泰国-中国\",\"orderStatus\":\"已出票\",\"operator\":\"曹坤祥\",\"purchaseTime\":\"2018-10-24 19:01:39\",\"orderValueAdded\":\" \",\"orderValueAddedSpecific\":\" \",\"remark\":\" \",\"officalReceivedPaymentStatus\":\" \",\"officalActualExpendAmount\":0,\"officalBalance\":0,\"officalRateAdjustDate\":null,\"passengerNames\":null,\"adultCount\":1,\"childCount\":0,\"sortIndex\":438}],\"info\":null}";
        if (StringUtils.isBlank(result)) {
//            return Response.error("获取数据失败");
            noteResult.setStatus(300);
            noteResult.setMsg("获取数据失败!");
            noteResult.setData(new HashMap<String, String>());
            return noteResult;
        }
        noteResult = JSONObject.parseObject(result, NoteResult.class);
        if(noteResult.getStatus() != 200){
            logger.error("[AutoSource] 从saleDetail表中查询报表数据失败，keyList="+keyList);
            noteResult.setMsg("[AutoSource] 从saleDetail表中查询报表数据失败，key="+keyList);
            noteResult.setStatus(300); // 获取失败状态值
            return noteResult;
        }
        // 从分页对象中取出 data ，再从 data.list 中取出 销售明细数据的数组
        List<SaleDetail> saleDetails = JSONArray.parseArray(JSONArray.toJSONString((noteResult.getData())), SaleDetail.class);
//        List<SaleDetail> saleDetails = JSONArray.parseArray(JSONArray.toJSONString(((PageInfo)noteResult.getData()).getList()), SaleDetail.class);
        logger.info("[AutoSource] 获取数据库表数据，总用时："+(System.currentTimeMillis()-beginTime)+"ms");
        try{
//            String customerId = keyList.substring(0,keyList.indexOf("_"));
//            String customerId = JSONObject.parseObject(requestBody).getString("customerId");
//            final String filePath =  saleDetailReportExcelPath + customerId + "/SalesDetailReport/";//指定文件保存路径

            // 使用浏览器的下载机制，而不是指定本地保存下载路径
            String path = parseToSaleDetail.createSaleDetailReport(saleDetails, response);
            noteResult.setStatus(Response.Status.SUCCEED.getStatus());
            noteResult.setData(path);
            noteResult.setMsg("[AutoSource] 生成报表成功，总用时:"+(System.currentTimeMillis()-beginTime)+"ms");
            logger.info("[AutoSource] 生成销售明细报表，总用时："+(System.currentTimeMillis()-beginTime)+"ms");
            return noteResult;
        }catch (Exception e){
            logger.error("[AutoSource] 生成报表发生异常",e);
            noteResult.setMsg("[AutoSource] 生成报表发生异常");
            return noteResult;
        }
    }





















    //任务消息队列，元素 调用AirTicket工程刷新SaleDetail表数据 的参数Map<String,String>
    public static BlockingQueue<Map<String, String>> updateSaleDetailQueue = new LinkedBlockingDeque<>();

    //提供入队列方法
    public static void setUpdateSaleDetailQueue(Map<String, String> paramsMap) {
        try {
            updateSaleDetailQueue.put(paramsMap);
        } catch (Exception e) {
            logger.error("updateSaleDetailQueue 队列,入队列异常,参数:" + paramsMap);
        }
    }

    /**
     * 采购信息更新时，更新报表数据，以队列方式执行
     *
     * @param requestBody
     * @return
     */
    public Response updateSaleDetail(String requestBody) {
        System.out.println(requestBody);
        if (StringUtils.isBlank(requestBody)) {
            return Response.error("[updateSaleDetail] 参数不合法,requestBody=null");
        }
        JSONObject reqObj = JSONObject.parseObject(requestBody);
        String customerId = reqObj.getString("customerId");
        String startTime = reqObj.getString("startTime");
        String endTime = reqObj.getString("endTime");
        if (StringUtils.isBlank(customerId) || StringUtils.isBlank(startTime) || StringUtils.isBlank(endTime)) {
            return Response.error("[updateSaleDetail] 参数不合法");
        }
        try {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            Date endDate = sdf.parse(endTime);
            if (endDate.getTime() > new Date().getTime()) {
                return Response.error("[updateSaleDetail] 当天订单采购信息修改不触发刷新报表数据，endTime=" + endTime);
            }
            Map<String, String> paramsMap = new HashMap<>();
            paramsMap.put("customerId", customerId);
            paramsMap.put("startTime", startTime);
            paramsMap.put("endTime", endTime);
            setUpdateSaleDetailQueue(paramsMap);//入队列
            logger.info("[updateSaleDetail] 采购信息变动，刷新报表数据任务已加入队列，参数：" + paramsMap);
            logger.info("[updateSaleDetail] 当前队列长度 updateSaleDetailQueue.size()=" + updateSaleDetailQueue.size());
            return Response.success("[updateSaleDetail] 已加入队列");
        } catch (Exception e) {
            logger.error("updateSaleDetail 发生异常", e);
        }
        return Response.error("未知原因失败");
    }

    @Override
    public void run(String... args) throws Exception {
        consumeUpdateSaleDetailQueue();
    }

    //消费响应消息队列
    private void consumeUpdateSaleDetailQueue() {
        ExecutorService threadPool = Executors.newFixedThreadPool(1);
        Runnable consumeTask = new Runnable() {
            @Override
            public void run() {
                while (true) {
                    logger.info("[updateSaleDetailQueue] 刷新报表队列 当前长度：" + updateSaleDetailQueue.size());
                    Map<String, String> map = null;
                    try {
                        //调用AirTicket工程刷新SaleDetail表数据
                        map = updateSaleDetailQueue.take();
                        HttpsSendData.send(overseasAirTicketUrl + "autoSaleDetai/updateSaleDetail.do", map);
                        logger.info("刷新报表已执行，参数：" + map);
                    } catch (Exception e) {
                        logger.error("刷新报表队列执行任务时异常,参数" + map);
                    }
                }
            }
        };
        threadPool.execute(consumeTask);
    }
}
